# Layers API Reference

## Making layers kernel-aware

### use_kernel_forward_from_hub[[kernels.use_kernel_forward_from_hub]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>kernels.use_kernel_forward_from_hub</name><anchor>kernels.use_kernel_forward_from_hub</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L988</source><parameters>[{"name": "layer_name", "val": ": str"}]</parameters><paramsdesc>- **layer_name** (`str`) --
  The name of the layer to use for kernel lookup in registered mappings.</paramsdesc><paramgroups>0</paramgroups><rettype>`Callable`</rettype><retdesc>A decorator function that can be applied to layer classes.</retdesc></docstring>

Decorator factory that makes a layer extensible using the specified layer name.

This is a decorator factory that returns a decorator which prepares a layer class to use kernels from the
Hugging Face Hub.







<ExampleCodeBlock anchor="kernels.use_kernel_forward_from_hub.example">

Example:
```python
import torch
import torch.nn as nn

from kernels import use_kernel_forward_from_hub
from kernels import Mode, kernelize

@use_kernel_forward_from_hub("MyCustomLayer")
class MyCustomLayer(nn.Module):
    def __init__(self, hidden_size):
        super().__init__()
        self.hidden_size = hidden_size

    def forward(self, x: torch.Tensor):
        # original implementation
        return x

model = MyCustomLayer(768)

# The layer can now be kernelized:
# model = kernelize(model, mode=Mode.TRAINING | Mode.TORCH_COMPILE, device="cuda")
```

</ExampleCodeBlock>


</div>

### replace_kernel_forward_from_hub[[kernels.replace_kernel_forward_from_hub]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>kernels.replace_kernel_forward_from_hub</name><anchor>kernels.replace_kernel_forward_from_hub</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L753</source><parameters>[{"name": "layer_name", "val": ": str"}]</parameters></docstring>

Function that prepares a layer class to use kernels from the Hugging Face Hub.

It is recommended to use [use_kernel_forward_from_hub()](/docs/kernels/pr_181/en/api/layers#kernels.use_kernel_forward_from_hub) decorator instead.
This function should only be used as a last resort to extend third-party layers,
it is inherently fragile since the member variables and `forward` signature
of usch a layer can change.

<ExampleCodeBlock anchor="kernels.replace_kernel_forward_from_hub.example">

Example:
```python
from kernels import replace_kernel_forward_from_hub
import torch.nn as nn

replace_kernel_forward_from_hub(nn.LayerNorm, "LayerNorm")
```

</ExampleCodeBlock>


</div>

## Registering kernel mappings

### use_kernel_mapping[[kernels.use_kernel_mapping]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>kernels.use_kernel_mapping</name><anchor>kernels.use_kernel_mapping</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L596</source><parameters>[{"name": "mapping", "val": ": Dict[str, Dict[Union[Device, str], Union[LayerRepositoryProtocol, Dict[Mode, LayerRepositoryProtocol]]]]"}, {"name": "inherit_mapping", "val": ": bool = True"}]</parameters><paramsdesc>- **mapping** (`Dict[str, Dict[Union[Device, str], Union[LayerRepositoryProtocol, Dict[Mode, LayerRepositoryProtocol]]]]`) --
  The kernel mapping to apply. Maps layer names to device-specific kernel configurations.
- **inherit_mapping** (`bool`, *optional*, defaults to `True`) --
  When `True`, the current mapping will be extended by `mapping` inside the context. When `False`,
  only `mapping` is used inside the context.</paramsdesc><paramgroups>0</paramgroups><retdesc>Context manager that handles the temporary kernel mapping.</retdesc></docstring>

Context manager that sets a kernel mapping for the duration of the context.

This function allows temporary kernel mappings to be applied within a specific context, enabling different
kernel configurations for different parts of your code.





<ExampleCodeBlock anchor="kernels.use_kernel_mapping.example">

Example:
```python
import torch
import torch.nn as nn
from torch.nn import functional as F

from kernels import use_kernel_forward_from_hub
from kernels import use_kernel_mapping, LayerRepository, Device
from kernels import Mode, kernelize

# Define a mapping
mapping = {
    "SiluAndMul": {
        "cuda": LayerRepository(
            repo_id="kernels-community/activation",
            layer_name="SiluAndMul",
        )
    }
}

@use_kernel_forward_from_hub("SiluAndMul")
class SiluAndMul(nn.Module):
    def forward(self, x: torch.Tensor) -> torch.Tensor:
        d = x.shape[-1] // 2
        return F.silu(x[..., :d]) * x[..., d:]

model = SiluAndMul()

# Use the mapping for the duration of the context.
with use_kernel_mapping(mapping):
    # kernelize uses the temporary mapping
    model = kernelize(model, mode=Mode.TRAINING | Mode.TORCH_COMPILE, device="cuda")

# Outside the context, original mappings are restored
```

</ExampleCodeBlock>


</div>

### register_kernel_mapping[[kernels.register_kernel_mapping]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>kernels.register_kernel_mapping</name><anchor>kernels.register_kernel_mapping</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L675</source><parameters>[{"name": "mapping", "val": ": Dict[str, Dict[Union[Device, str], Union[LayerRepositoryProtocol, Dict[Mode, LayerRepositoryProtocol]]]]"}, {"name": "inherit_mapping", "val": ": bool = True"}]</parameters><paramsdesc>- **mapping** (`Dict[str, Dict[Union[Device, str], Union[LayerRepositoryProtocol, Dict[Mode, LayerRepositoryProtocol]]]]`) --
  The kernel mapping to register globally. Maps layer names to device-specific kernels.
  The mapping can specify different kernels for different modes (training, inference, etc.).
- **inherit_mapping** (`bool`, *optional*, defaults to `True`) --
  When `True`, the current mapping will be extended by `mapping`. When `False`, the existing mappings
  are erased before adding `mapping`.</paramsdesc><paramgroups>0</paramgroups></docstring>

Register a global mapping between layer names and their corresponding kernel implementations.

This function allows you to register a mapping between a layer name and the corresponding kernel(s) to use,
depending on the device and mode. This should be used in conjunction with [kernelize()](/docs/kernels/pr_181/en/api/layers#kernels.kernelize).



<ExampleCodeBlock anchor="kernels.register_kernel_mapping.example">

Example:
```python
from kernels import LayerRepository, register_kernel_mapping, Mode

# Simple mapping for a single kernel per device
kernel_layer_mapping = {
    "LlamaRMSNorm": {
        "cuda": LayerRepository(
            repo_id="kernels-community/activation",
            layer_name="RmsNorm",
            revision="layers",
        ),
    },
}
register_kernel_mapping(kernel_layer_mapping)

# Advanced mapping with mode-specific kernels
advanced_mapping = {
    "MultiHeadAttention": {
        "cuda": {
            Mode.TRAINING: LayerRepository(
                repo_id="username/training-kernels",
                layer_name="TrainingAttention"
            ),
            Mode.INFERENCE: LayerRepository(
                repo_id="username/inference-kernels",
                layer_name="FastAttention"
            ),
        }
    }
}
register_kernel_mapping(advanced_mapping)
```

</ExampleCodeBlock>


</div>

## Kernelizing a model

### kernelize[[kernels.kernelize]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>kernels.kernelize</name><anchor>kernels.kernelize</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L822</source><parameters>[{"name": "model", "val": ": 'nn.Module'"}, {"name": "mode", "val": ": Mode"}, {"name": "device", "val": ": Optional[Union[str, 'torch.device']] = None"}, {"name": "use_fallback", "val": ": bool = True"}]</parameters><paramsdesc>- **model** (`nn.Module`) --
  The PyTorch model to kernelize.
- **mode** ([Mode](/docs/kernels/pr_181/en/api/layers#kernels.Mode)) -- The mode that the kernel is going to be used in. For example,
  `Mode.TRAINING | Mode.TORCH_COMPILE` kernelizes the model for training with
  `torch.compile`.
- **device** (`Union[str, torch.device]`, *optional*) --
  The device type to load kernels for. Supported device types are: "cuda", "mps", "npu", "rocm", "xpu".
  The device type will be inferred from the model parameters when not provided.
- **use_fallback** (`bool`, *optional*, defaults to `True`) --
  Whether to use the original forward method of modules when no compatible kernel could be found.
  If set to `False`, an exception will be raised in such cases.</paramsdesc><paramgroups>0</paramgroups><rettype>`nn.Module`</rettype><retdesc>The kernelized model with optimized kernel implementations.</retdesc></docstring>

Replace layer forward methods with optimized kernel implementations.

This function iterates over all modules in the model and replaces the `forward` method of extensible layers
for which kernels are registered using [register_kernel_mapping()](/docs/kernels/pr_181/en/api/layers#kernels.register_kernel_mapping) or [use_kernel_mapping()](/docs/kernels/pr_181/en/api/layers#kernels.use_kernel_mapping).







<ExampleCodeBlock anchor="kernels.kernelize.example">

Example:
```python
import torch
import torch.nn as nn

from kernels import kernelize, Mode, register_kernel_mapping, LayerRepository
from kernels import use_kernel_forward_from_hub

@use_kernel_forward_from_hub("SiluAndMul")
class SiluAndMul(nn.Module):
    def forward(self, x: torch.Tensor) -> torch.Tensor:
        d = x.shape[-1] // 2
        return F.silu(x[..., :d]) * x[..., d:]

mapping = {
    "SiluAndMul": {
        "cuda": LayerRepository(
            repo_id="kernels-community/activation",
            layer_name="SiluAndMul",
        )
    }
}
register_kernel_mapping(mapping)

# Create and kernelize a model
model = nn.Sequential(
    nn.Linear(1024, 2048, device="cuda"),
    SiluAndMul(),
)

# Kernelize for inference
kernelized_model = kernelize(model, mode=Mode.TRAINING | Mode.TORCH_COMPILE)
```

</ExampleCodeBlock>


</div>

## Classes

### Device[[kernels.Device]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class kernels.Device</name><anchor>kernels.Device</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L81</source><parameters>[{"name": "type", "val": ": str"}, {"name": "properties", "val": ": Optional[CUDAProperties] = None"}]</parameters><paramsdesc>- **type** (`str`) --
  The device type (e.g., "cuda", "mps", "npu", "rocm", "xpu").
- **properties** (`CUDAProperties`, *optional*) --
  Device-specific properties. Currently only `CUDAProperties` is supported for CUDA devices.</paramsdesc><paramgroups>0</paramgroups></docstring>

Represents a compute device with optional properties.

This class encapsulates device information including device type and optional device-specific properties
like CUDA capabilities.



<ExampleCodeBlock anchor="kernels.Device.example">

Example:
```python
from kernels import Device, CUDAProperties

# Basic CUDA device
cuda_device = Device(type="cuda")

# CUDA device with specific capability requirements
cuda_device_with_props = Device(
    type="cuda",
    properties=CUDAProperties(min_capability=75, max_capability=90)
)

# MPS device for Apple Silicon
mps_device = Device(type="mps")

# XPU device (e.g., Intel(R) Data Center GPU Max 1550)
xpu_device = Device(type="xpu")

# NPU device (Huawei Ascend)
npu_device = Device(type="npu")
```

</ExampleCodeBlock>



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_repo</name><anchor>kernels.Device.create_repo</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L126</source><parameters>[]</parameters></docstring>
Create an appropriate repository set for this device type.

</div></div>

### Mode[[kernels.Mode]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class kernels.Mode</name><anchor>kernels.Mode</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L43</source><parameters>[{"name": "value", "val": ""}, {"name": "names", "val": " = None"}, {"name": "module", "val": " = None"}, {"name": "qualname", "val": " = None"}, {"name": "type", "val": " = None"}, {"name": "start", "val": " = 1"}]</parameters><paramsdesc>- **INFERENCE** -- The kernel is used for inference.
- **TRAINING** -- The kernel is used for training.
- **TORCH_COMPILE** -- The kernel is used with `torch.compile`.
- **FALLBACK** -- In a kernel mapping, this kernel is used when no other mode matches.</paramsdesc><paramgroups>0</paramgroups></docstring>

Kernelize mode

The `Mode` flag is used by [kernelize()](/docs/kernels/pr_181/en/api/layers#kernels.kernelize) to select kernels for the given mode. Mappings can be registered for
specific modes.



Note:
Different modes can be combined. For instance, `INFERENCE | TORCH_COMPILE` should be used for layers that
are used for inference *with* `torch.compile`.



</div>

### LayerRepository[[kernels.LayerRepository]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class kernels.LayerRepository</name><anchor>kernels.LayerRepository</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L247</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "layer_name", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "version", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The Hub repository containing the layer.
- **layer_name** (`str`) --
  The name of the layer within the kernel repository.
- **revision** (`str`, *optional*, defaults to `"main"`) --
  The specific revision (branch, tag, or commit) to download. Cannot be used together with `version`.
- **version** (`str`, *optional*) --
  The kernel version to download. This can be a Python version specifier, such as `">=1.0.0,<2.0.0"`.
  Cannot be used together with `revision`.</paramsdesc><paramgroups>0</paramgroups></docstring>

Repository and name of a layer for kernel mapping.



<ExampleCodeBlock anchor="kernels.LayerRepository.example">

Example:
```python
from kernels import LayerRepository

# Reference a specific layer by revision
layer_repo = LayerRepository(
    repo_id="kernels-community/activation",
    layer_name="SiluAndMul",
)

# Reference a layer by version constraint
layer_repo_versioned = LayerRepository(
    repo_id="kernels-community/activation",
    layer_name="SiluAndMul",
    version=">=0.0.3,<0.1"
)
```

</ExampleCodeBlock>


</div>

### LocalLayerRepository[[kernels.LocalLayerRepository]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class kernels.LocalLayerRepository</name><anchor>kernels.LocalLayerRepository</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L327</source><parameters>[{"name": "repo_path", "val": ": Path"}, {"name": "package_name", "val": ": str"}, {"name": "layer_name", "val": ": str"}]</parameters><paramsdesc>- **repo_path** (`Path`) --
  The local repository containing the layer.
- **package_name** (`str`) --
  Package name of the kernel.
- **layer_name** (`str`) --
  The name of the layer within the kernel repository.</paramsdesc><paramgroups>0</paramgroups></docstring>

Repository from a local directory for kernel mapping.



<ExampleCodeBlock anchor="kernels.LocalLayerRepository.example">

Example:
```python
from pathlib import Path

from kernels import LocalLayerRepository

# Reference a specific layer by revision
layer_repo = LocalLayerRepository(
    repo_path=Path("/home/daniel/kernels/activation"),
    package_name="activation",
    layer_name="SiluAndMul",
)
```

</ExampleCodeBlock>


</div>

### LockedLayerRepository[[kernels.LockedLayerRepository]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class kernels.LockedLayerRepository</name><anchor>kernels.LockedLayerRepository</anchor><source>https://github.com/huggingface/kernels/blob/vr_181/src/kernels/layer.py#L383</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "lockfile", "val": ": Optional[Path] = None"}, {"name": "layer_name", "val": ": str"}]</parameters></docstring>

Repository and name of a layer.

In contrast to `LayerRepository`, this class uses repositories that
are locked inside a project.


</div>

<EditOnGithub source="https://github.com/huggingface/kernels/blob/main/docs/source/api/layers.md" />