import{s as mt,n as ht,o as dt}from"../chunks/scheduler.65852ee5.js";import{S as bt,i as It,g as M,s as n,r as i,A as gt,h as p,f as s,c as a,j as Tt,u as y,x as o,k as wt,y as Ct,a as e,v as c,d as j,t as r,w as u}from"../chunks/index.aa74147d.js";import{D as ft,C as J}from"../chunks/DocNotebookDropdown.479f4286.js";import{H as Yl,E as At}from"../chunks/EditOnGithub.4eda6a96.js";function Bt(Fl){let U,nl,sl,al,T,Ml,w,pl,m,Ll='<em>Authored by: <a href="https://huggingface.co/m-ric" rel="nofollow">Aymeric Roucher</a></em>',ol,h,zl="In this tutorial, we’ll see how to implement an agent that leverages SQL using <code>transformers.agents</code>.",il,d,Dl="What’s the advantage over a standard text-to-SQL pipeline?",yl,b,Kl="A standard text-to-sql pipeline is brittle, since the generated SQL query can be incorrect. Even worse, the query could be incorrect, but not raise an error, instead giving some incorrect/useless outputs without raising an alarm.",cl,I,Pl="👉 Instead, <strong>an agent system is able to critically inspect outputs and decide if the query needs to be changed or not</strong>, thus giving it a huge performance boost.",jl,g,Ol="Let’s build this agent! 💪",rl,C,ul,f,Jl,A,Ul,B,lt="Let’s check that our system works with a basic query:",Tl,q,wl,W,tt=`(1, 'Alan Payne', 12.06, 1.2)
(2, 'Alex Mason', 23.86, 0.24)
(3, 'Woodrow Wilson', 53.43, 5.43)
(4, 'Margaret James', 21.11, 1.0)
`,ml,Q,hl,Z,st="Now let’s make our SQL table retrievable by a tool.",dl,E,et="The tool’s <code>description</code> attribute will be embedded in the LLM’s prompt by the agent system: it gives the LLM information about how to use the tool. So that is where we want to describe the SQL table.",bl,R,Il,N,nt=`Columns:
  - receipt_id: INTEGER
  - customer_name: VARCHAR(16)
  - price: FLOAT
  - tip: FLOAT
`,gl,k,at='Now the tool will need a few mandatory arguments (read <a href="https://huggingface.co/docs/transformers/en/agents" rel="nofollow">the documentation</a> for more detail):',Cl,v,Mt="<li>The <code>description</code> mentioned above</li> <li>A <code>name</code> by which the agent system can call it.</li> <li>A dictionary of <code>inputs</code>: describe the tool’s input, each with a <code>type</code> and a <code>description</code> key.</li> <li>An <code>output_type</code> for compatibility with gradio demos for instance.</li> <li>A <code>forward</code> method that gets called when the tool is called.</li>",fl,S,Al,V,pt="Now let us create an agent that leverages this tool.",Bl,_,ot="We use the <code>ReactCodeAgent</code>, which is <code>transformers.agents</code>’ main agent class: an agent that writes actions in code and can iterate on previous output according to the ReAct framework.",ql,G,it='The <code>llm_engine</code> is the LLM that powers the agent system. <code>HfEngine</code> allows you to call LLMs using HF’s Inference API, either via Serverless or Dedicated endpoint, but you could also use any proprietary API: check out <a href="agent_change_llm">this other cookbook</a> to learn how to adapt it.',Wl,$,Ql,x,Zl,X,El,H,yt="Now let’s make it more challenging! We want our agent to handle joins across multiple tables.",Rl,Y,ct="So let’s make a second table recording the names of waiters for each <code>receipt_id</code>!",Nl,F,kl,L,jt="We need to update the <code>SQLExecutorTool</code> with this table’s description to let the LLM properly leverage information from this table.",vl,z,Sl,D,rt=`Allows you to perform SQL queries on the table. Beware that this tool's output is a string representation of the execution output.
It can use the following tables:

Table 'receipts':
Columns:
  - receipt_id: INTEGER
  - customer_name: VARCHAR(16)
  - price: FLOAT
  - tip: FLOAT

Table 'waiters':
Columns:
  - receipt_id: INTEGER
  - waiter_name: VARCHAR(16)
`,Vl,K,_l,P,ut=`['time', 'queue', 'random', 'unicodedata', 're', 'collections', 'itertools', 'stat', 'math', 'statistics']
`,Gl,O,Jt="It directly works! The setup was surprisingly simple, wasn’t it?",$l,ll,Ut="✅ Now you can go build this text-to-SQL system you’ve always dreamt of! ✨",xl,tl,Xl,el,Hl;return T=new ft({props:{classNames:"absolute z-10 right-0 top-0",options:[{label:"Google Colab",value:"https://colab.research.google.com/github/huggingface/cookbook/blob/enterprise-cookbook-gradio-fixes/notebooks/en/agent_text_to_sql.ipynb"}]}}),w=new Yl({props:{title:"Agent for text-to-SQL with automatic error correction",local:"agent-for-text-to-sql-with-automatic-error-correction",headingTag:"h1"}}),C=new Yl({props:{title:"Setup SQL tables",local:"setup-sql-tables",headingTag:"h2"}}),f=new J({props:{code:"ZnJvbSUyMHNxbGFsY2hlbXklMjBpbXBvcnQlMjAoJTBBJTIwJTIwJTIwJTIwY3JlYXRlX2VuZ2luZSUyQyUwQSUyMCUyMCUyMCUyME1ldGFEYXRhJTJDJTBBJTIwJTIwJTIwJTIwVGFibGUlMkMlMEElMjAlMjAlMjAlMjBDb2x1bW4lMkMlMEElMjAlMjAlMjAlMjBTdHJpbmclMkMlMEElMjAlMjAlMjAlMjBJbnRlZ2VyJTJDJTBBJTIwJTIwJTIwJTIwRmxvYXQlMkMlMEElMjAlMjAlMjAlMjBpbnNlcnQlMkMlMEElMjAlMjAlMjAlMjBpbnNwZWN0JTJDJTBBJTIwJTIwJTIwJTIwdGV4dCUyQyUwQSklMEElMEFlbmdpbmUlMjAlM0QlMjBjcmVhdGVfZW5naW5lKCUyMnNxbGl0ZSUzQSUyRiUyRiUyRiUzQW1lbW9yeSUzQSUyMiklMEFtZXRhZGF0YV9vYmolMjAlM0QlMjBNZXRhRGF0YSgpJTBBJTBBJTIzJTIwY3JlYXRlJTIwY2l0eSUyMFNRTCUyMHRhYmxlJTBBdGFibGVfbmFtZSUyMCUzRCUyMCUyMnJlY2VpcHRzJTIyJTBBcmVjZWlwdHMlMjAlM0QlMjBUYWJsZSglMEElMjAlMjAlMjAlMjB0YWJsZV9uYW1lJTJDJTBBJTIwJTIwJTIwJTIwbWV0YWRhdGFfb2JqJTJDJTBBJTIwJTIwJTIwJTIwQ29sdW1uKCUyMnJlY2VpcHRfaWQlMjIlMkMlMjBJbnRlZ2VyJTJDJTIwcHJpbWFyeV9rZXklM0RUcnVlKSUyQyUwQSUyMCUyMCUyMCUyMENvbHVtbiglMjJjdXN0b21lcl9uYW1lJTIyJTJDJTIwU3RyaW5nKDE2KSUyQyUyMHByaW1hcnlfa2V5JTNEVHJ1ZSklMkMlMEElMjAlMjAlMjAlMjBDb2x1bW4oJTIycHJpY2UlMjIlMkMlMjBGbG9hdCklMkMlMEElMjAlMjAlMjAlMjBDb2x1bW4oJTIydGlwJTIyJTJDJTIwRmxvYXQpJTJDJTBBKSUwQW1ldGFkYXRhX29iai5jcmVhdGVfYWxsKGVuZ2luZSk=",highlighted:`<span class="hljs-keyword">from</span> sqlalchemy <span class="hljs-keyword">import</span> (
    create_engine,
    MetaData,
    Table,
    Column,
    String,
    Integer,
    Float,
    insert,
    inspect,
    text,
)

engine = create_engine(<span class="hljs-string">&quot;sqlite:///:memory:&quot;</span>)
metadata_obj = MetaData()

<span class="hljs-comment"># create city SQL table</span>
table_name = <span class="hljs-string">&quot;receipts&quot;</span>
receipts = Table(
    table_name,
    metadata_obj,
    Column(<span class="hljs-string">&quot;receipt_id&quot;</span>, Integer, primary_key=<span class="hljs-literal">True</span>),
    Column(<span class="hljs-string">&quot;customer_name&quot;</span>, String(<span class="hljs-number">16</span>), primary_key=<span class="hljs-literal">True</span>),
    Column(<span class="hljs-string">&quot;price&quot;</span>, Float),
    Column(<span class="hljs-string">&quot;tip&quot;</span>, Float),
)
metadata_obj.create_all(engine)`,wrap:!1}}),A=new J({props:{code:"cm93cyUyMCUzRCUyMCU1QiUwQSUyMCUyMCUyMCUyMCU3QiUyMnJlY2VpcHRfaWQlMjIlM0ElMjAxJTJDJTIwJTIyY3VzdG9tZXJfbmFtZSUyMiUzQSUyMCUyMkFsYW4lMjBQYXluZSUyMiUyQyUyMCUyMnByaWNlJTIyJTNBJTIwMTIuMDYlMkMlMjAlMjJ0aXAlMjIlM0ElMjAxLjIwJTdEJTJDJTBBJTIwJTIwJTIwJTIwJTdCJTIycmVjZWlwdF9pZCUyMiUzQSUyMDIlMkMlMjAlMjJjdXN0b21lcl9uYW1lJTIyJTNBJTIwJTIyQWxleCUyME1hc29uJTIyJTJDJTIwJTIycHJpY2UlMjIlM0ElMjAyMy44NiUyQyUyMCUyMnRpcCUyMiUzQSUyMDAuMjQlN0QlMkMlMEElMjAlMjAlMjAlMjAlN0IlMjJyZWNlaXB0X2lkJTIyJTNBJTIwMyUyQyUyMCUyMmN1c3RvbWVyX25hbWUlMjIlM0ElMjAlMjJXb29kcm93JTIwV2lsc29uJTIyJTJDJTIwJTIycHJpY2UlMjIlM0ElMjA1My40MyUyQyUyMCUyMnRpcCUyMiUzQSUyMDUuNDMlN0QlMkMlMEElMjAlMjAlMjAlMjAlN0IlMjJyZWNlaXB0X2lkJTIyJTNBJTIwNCUyQyUyMCUyMmN1c3RvbWVyX25hbWUlMjIlM0ElMjAlMjJNYXJnYXJldCUyMEphbWVzJTIyJTJDJTIwJTIycHJpY2UlMjIlM0ElMjAyMS4xMSUyQyUyMCUyMnRpcCUyMiUzQSUyMDEuMDAlN0QlMkMlMEElNUQlMEFmb3IlMjByb3clMjBpbiUyMHJvd3MlM0ElMEElMjAlMjAlMjAlMjBzdG10JTIwJTNEJTIwaW5zZXJ0KHJlY2VpcHRzKS52YWx1ZXMoKipyb3cpJTBBJTIwJTIwJTIwJTIwd2l0aCUyMGVuZ2luZS5iZWdpbigpJTIwYXMlMjBjb25uZWN0aW9uJTNBJTBBJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwY3Vyc29yJTIwJTNEJTIwY29ubmVjdGlvbi5leGVjdXRlKHN0bXQp",highlighted:`rows = [
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">1</span>, <span class="hljs-string">&quot;customer_name&quot;</span>: <span class="hljs-string">&quot;Alan Payne&quot;</span>, <span class="hljs-string">&quot;price&quot;</span>: <span class="hljs-number">12.06</span>, <span class="hljs-string">&quot;tip&quot;</span>: <span class="hljs-number">1.20</span>},
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">2</span>, <span class="hljs-string">&quot;customer_name&quot;</span>: <span class="hljs-string">&quot;Alex Mason&quot;</span>, <span class="hljs-string">&quot;price&quot;</span>: <span class="hljs-number">23.86</span>, <span class="hljs-string">&quot;tip&quot;</span>: <span class="hljs-number">0.24</span>},
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">3</span>, <span class="hljs-string">&quot;customer_name&quot;</span>: <span class="hljs-string">&quot;Woodrow Wilson&quot;</span>, <span class="hljs-string">&quot;price&quot;</span>: <span class="hljs-number">53.43</span>, <span class="hljs-string">&quot;tip&quot;</span>: <span class="hljs-number">5.43</span>},
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">4</span>, <span class="hljs-string">&quot;customer_name&quot;</span>: <span class="hljs-string">&quot;Margaret James&quot;</span>, <span class="hljs-string">&quot;price&quot;</span>: <span class="hljs-number">21.11</span>, <span class="hljs-string">&quot;tip&quot;</span>: <span class="hljs-number">1.00</span>},
]
<span class="hljs-keyword">for</span> row <span class="hljs-keyword">in</span> rows:
    stmt = insert(receipts).values(**row)
    <span class="hljs-keyword">with</span> engine.begin() <span class="hljs-keyword">as</span> connection:
        cursor = connection.execute(stmt)`,wrap:!1}}),q=new J({props:{code:"d2l0aCUyMGVuZ2luZS5jb25uZWN0KCklMjBhcyUyMGNvbiUzQSUwQSUyMCUyMCUyMCUyMHJvd3MlMjAlM0QlMjBjb24uZXhlY3V0ZSh0ZXh0KCUyMiUyMiUyMlNFTEVDVCUyMColMjBmcm9tJTIwcmVjZWlwdHMlMjIlMjIlMjIpKSUwQSUyMCUyMCUyMCUyMGZvciUyMHJvdyUyMGluJTIwcm93cyUzQSUwQSUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMHByaW50KHJvdyk=",highlighted:`<span class="hljs-meta">&gt;&gt;&gt; </span><span class="hljs-keyword">with</span> engine.connect() <span class="hljs-keyword">as</span> con:
<span class="hljs-meta">... </span>    rows = con.execute(text(<span class="hljs-string">&quot;&quot;&quot;SELECT * from receipts&quot;&quot;&quot;</span>))
<span class="hljs-meta">... </span>    <span class="hljs-keyword">for</span> row <span class="hljs-keyword">in</span> rows:
<span class="hljs-meta">... </span>        <span class="hljs-built_in">print</span>(row)`,wrap:!1}}),Q=new Yl({props:{title:"Build our agent",local:"build-our-agent",headingTag:"h2"}}),R=new J({props:{code:"aW5zcGVjdG9yJTIwJTNEJTIwaW5zcGVjdChlbmdpbmUpJTBBY29sdW1uc19pbmZvJTIwJTNEJTIwJTVCKGNvbCU1QiUyMm5hbWUlMjIlNUQlMkMlMjBjb2wlNUIlMjJ0eXBlJTIyJTVEKSUyMGZvciUyMGNvbCUyMGluJTIwaW5zcGVjdG9yLmdldF9jb2x1bW5zKCUyMnJlY2VpcHRzJTIyKSU1RCUwQSUwQXRhYmxlX2Rlc2NyaXB0aW9uJTIwJTNEJTIwJTIyQ29sdW1ucyUzQSU1Q24lMjIlMjAlMkIlMjAlMjIlNUNuJTIyLmpvaW4oJTVCZiUyMiUyMCUyMC0lMjAlN0JuYW1lJTdEJTNBJTIwJTdCY29sX3R5cGUlN0QlMjIlMjBmb3IlMjBuYW1lJTJDJTIwY29sX3R5cGUlMjBpbiUyMGNvbHVtbnNfaW5mbyU1RCklMEFwcmludCh0YWJsZV9kZXNjcmlwdGlvbik=",highlighted:`<span class="hljs-meta">&gt;&gt;&gt; </span>inspector = inspect(engine)
<span class="hljs-meta">&gt;&gt;&gt; </span>columns_info = [(col[<span class="hljs-string">&quot;name&quot;</span>], col[<span class="hljs-string">&quot;type&quot;</span>]) <span class="hljs-keyword">for</span> col <span class="hljs-keyword">in</span> inspector.get_columns(<span class="hljs-string">&quot;receipts&quot;</span>)]

<span class="hljs-meta">&gt;&gt;&gt; </span>table_description = <span class="hljs-string">&quot;Columns:\\n&quot;</span> + <span class="hljs-string">&quot;\\n&quot;</span>.join([<span class="hljs-string">f&quot;  - <span class="hljs-subst">{name}</span>: <span class="hljs-subst">{col_type}</span>&quot;</span> <span class="hljs-keyword">for</span> name, col_type <span class="hljs-keyword">in</span> columns_info])
<span class="hljs-meta">&gt;&gt;&gt; </span><span class="hljs-built_in">print</span>(table_description)`,wrap:!1}}),S=new J({props:{code:"ZnJvbSUyMHRyYW5zZm9ybWVycy5hZ2VudHMlMjBpbXBvcnQlMjBUb29sJTBBJTBBJTBBY2xhc3MlMjBTUUxFeGVjdXRvclRvb2woVG9vbCklM0ElMEElMjAlMjAlMjAlMjBuYW1lJTIwJTNEJTIwJTIyc3FsX2VuZ2luZSUyMiUwQSUyMCUyMCUyMCUyMGRlc2NyaXB0aW9uJTIwJTNEJTIwZiUyMiUyMiUyMkFsbG93cyUyMHlvdSUyMHRvJTIwcGVyZm9ybSUyMFNRTCUyMHF1ZXJpZXMlMjBvbiUyMHRoZSUyMHRhYmxlLiUyMFJldHVybnMlMjBhJTIwc3RyaW5nJTIwcmVwcmVzZW50YXRpb24lMjBvZiUyMHRoZSUyMHJlc3VsdC4lMEFUaGUlMjB0YWJsZSUyMGlzJTIwbmFtZWQlMjAncmVjZWlwdHMnLiUyMEl0cyUyMGRlc2NyaXB0aW9uJTIwaXMlMjBhcyUyMGZvbGxvd3MlM0ElMjAlNUNuJTdCdGFibGVfZGVzY3JpcHRpb24lN0QlMjIlMjIlMjIlMEElMjAlMjAlMjAlMjBpbnB1dHMlMjAlM0QlMjAlN0IlMEElMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjJxdWVyeSUyMiUzQSUyMCU3QiUwQSUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMnR5cGUlMjIlM0ElMjAlMjJ0ZXh0JTIyJTJDJTBBJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIyZGVzY3JpcHRpb24lMjIlM0ElMjBmJTIyVGhlJTIwcXVlcnklMjB0byUyMHBlcmZvcm0uJTIwVGhpcyUyMHNob3VsZCUyMGJlJTIwY29ycmVjdCUyMFNRTC4lMjIlMkMlMEElMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlN0QlMEElMjAlMjAlMjAlMjAlN0QlMEElMjAlMjAlMjAlMjBvdXRwdXRfdHlwZSUyMCUzRCUyMCUyMnRleHQlMjIlMEElMEElMjAlMjAlMjAlMjBkZWYlMjBmb3J3YXJkKHNlbGYlMkMlMjBxdWVyeSUzQSUyMHN0ciklMjAtJTNFJTIwc3RyJTNBJTBBJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwb3V0cHV0JTIwJTNEJTIwJTIyJTIyJTBBJTIwJTIwJTIwJTIwJTIwJTIwJTIwJTIwd2l0aCUyMGVuZ2luZS5jb25uZWN0KCklMjBhcyUyMGNvbiUzQSUwQSUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMCUyMHJvd3MlMjAlM0QlMjBjb24uZXhlY3V0ZSh0ZXh0KHF1ZXJ5KSklMEElMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjBmb3IlMjByb3clMjBpbiUyMHJvd3MlM0ElMEElMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjBvdXRwdXQlMjAlMkIlM0QlMjAlMjIlNUNuJTIyJTIwJTJCJTIwc3RyKHJvdyklMEElMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjByZXR1cm4lMjBvdXRwdXQ=",highlighted:`<span class="hljs-keyword">from</span> transformers.agents <span class="hljs-keyword">import</span> Tool


<span class="hljs-keyword">class</span> <span class="hljs-title class_">SQLExecutorTool</span>(<span class="hljs-title class_ inherited__">Tool</span>):
    name = <span class="hljs-string">&quot;sql_engine&quot;</span>
    description = <span class="hljs-string">f&quot;&quot;&quot;Allows you to perform SQL queries on the table. Returns a string representation of the result.
The table is named &#x27;receipts&#x27;. Its description is as follows: \\n<span class="hljs-subst">{table_description}</span>&quot;&quot;&quot;</span>
    inputs = {
        <span class="hljs-string">&quot;query&quot;</span>: {
            <span class="hljs-string">&quot;type&quot;</span>: <span class="hljs-string">&quot;text&quot;</span>,
            <span class="hljs-string">&quot;description&quot;</span>: <span class="hljs-string">f&quot;The query to perform. This should be correct SQL.&quot;</span>,
        }
    }
    output_type = <span class="hljs-string">&quot;text&quot;</span>

    <span class="hljs-keyword">def</span> <span class="hljs-title function_">forward</span>(<span class="hljs-params">self, query: <span class="hljs-built_in">str</span></span>) -&gt; <span class="hljs-built_in">str</span>:
        output = <span class="hljs-string">&quot;&quot;</span>
        <span class="hljs-keyword">with</span> engine.connect() <span class="hljs-keyword">as</span> con:
            rows = con.execute(text(query))
            <span class="hljs-keyword">for</span> row <span class="hljs-keyword">in</span> rows:
                output += <span class="hljs-string">&quot;\\n&quot;</span> + <span class="hljs-built_in">str</span>(row)
        <span class="hljs-keyword">return</span> output`,wrap:!1}}),$=new J({props:{code:"ZnJvbSUyMHRyYW5zZm9ybWVycy5hZ2VudHMlMjBpbXBvcnQlMjBSZWFjdENvZGVBZ2VudCUyQyUyMEhmRW5naW5lJTBBJTBBYWdlbnQlMjAlM0QlMjBSZWFjdENvZGVBZ2VudCglMEElMjAlMjAlMjAlMjB0b29scyUzRCU1QlNRTEV4ZWN1dG9yVG9vbCgpJTVEJTJDJTBBJTIwJTIwJTIwJTIwbGxtX2VuZ2luZSUzREhmRW5naW5lKCUyMm1ldGEtbGxhbWElMkZNZXRhLUxsYW1hLTMtNzBCLUluc3RydWN0JTIyKSUyQyUwQSk=",highlighted:`<span class="hljs-keyword">from</span> transformers.agents <span class="hljs-keyword">import</span> ReactCodeAgent, HfEngine

agent = ReactCodeAgent(
    tools=[SQLExecutorTool()],
    llm_engine=HfEngine(<span class="hljs-string">&quot;meta-llama/Meta-Llama-3-70B-Instruct&quot;</span>),
)`,wrap:!1}}),x=new J({props:{code:"YWdlbnQucnVuKCUyMkNhbiUyMHlvdSUyMGdpdmUlMjBtZSUyMHRoZSUyMG5hbWUlMjBvZiUyMHRoZSUyMGNsaWVudCUyMHdobyUyMGdvdCUyMHRoZSUyMG1vc3QlMjBleHBlbnNpdmUlMjByZWNlaXB0JTNGJTIyKQ==",highlighted:'agent.run(<span class="hljs-string">&quot;Can you give me the name of the client who got the most expensive receipt?&quot;</span>)',wrap:!1}}),X=new Yl({props:{title:"Increasing difficulty: Table joins",local:"increasing-difficulty-table-joins",headingTag:"h2"}}),F=new J({props:{code:"dGFibGVfbmFtZSUyMCUzRCUyMCUyMndhaXRlcnMlMjIlMEFyZWNlaXB0cyUyMCUzRCUyMFRhYmxlKCUwQSUyMCUyMCUyMCUyMHRhYmxlX25hbWUlMkMlMEElMjAlMjAlMjAlMjBtZXRhZGF0YV9vYmolMkMlMEElMjAlMjAlMjAlMjBDb2x1bW4oJTIycmVjZWlwdF9pZCUyMiUyQyUyMEludGVnZXIlMkMlMjBwcmltYXJ5X2tleSUzRFRydWUpJTJDJTBBJTIwJTIwJTIwJTIwQ29sdW1uKCUyMndhaXRlcl9uYW1lJTIyJTJDJTIwU3RyaW5nKDE2KSUyQyUyMHByaW1hcnlfa2V5JTNEVHJ1ZSklMkMlMEEpJTBBbWV0YWRhdGFfb2JqLmNyZWF0ZV9hbGwoZW5naW5lKSUwQSUwQXJvd3MlMjAlM0QlMjAlNUIlMEElMjAlMjAlMjAlMjAlN0IlMjJyZWNlaXB0X2lkJTIyJTNBJTIwMSUyQyUyMCUyMndhaXRlcl9uYW1lJTIyJTNBJTIwJTIyQ29yZXklMjBKb2huc29uJTIyJTdEJTJDJTBBJTIwJTIwJTIwJTIwJTdCJTIycmVjZWlwdF9pZCUyMiUzQSUyMDIlMkMlMjAlMjJ3YWl0ZXJfbmFtZSUyMiUzQSUyMCUyMk1pY2hhZWwlMjBXYXR0cyUyMiU3RCUyQyUwQSUyMCUyMCUyMCUyMCU3QiUyMnJlY2VpcHRfaWQlMjIlM0ElMjAzJTJDJTIwJTIyd2FpdGVyX25hbWUlMjIlM0ElMjAlMjJNaWNoYWVsJTIwV2F0dHMlMjIlN0QlMkMlMEElMjAlMjAlMjAlMjAlN0IlMjJyZWNlaXB0X2lkJTIyJTNBJTIwNCUyQyUyMCUyMndhaXRlcl9uYW1lJTIyJTNBJTIwJTIyTWFyZ2FyZXQlMjBKYW1lcyUyMiU3RCUyQyUwQSU1RCUwQWZvciUyMHJvdyUyMGluJTIwcm93cyUzQSUwQSUyMCUyMCUyMCUyMHN0bXQlMjAlM0QlMjBpbnNlcnQocmVjZWlwdHMpLnZhbHVlcygqKnJvdyklMEElMjAlMjAlMjAlMjB3aXRoJTIwZW5naW5lLmJlZ2luKCklMjBhcyUyMGNvbm5lY3Rpb24lM0ElMEElMjAlMjAlMjAlMjAlMjAlMjAlMjAlMjBjdXJzb3IlMjAlM0QlMjBjb25uZWN0aW9uLmV4ZWN1dGUoc3RtdCk=",highlighted:`table_name = <span class="hljs-string">&quot;waiters&quot;</span>
receipts = Table(
    table_name,
    metadata_obj,
    Column(<span class="hljs-string">&quot;receipt_id&quot;</span>, Integer, primary_key=<span class="hljs-literal">True</span>),
    Column(<span class="hljs-string">&quot;waiter_name&quot;</span>, String(<span class="hljs-number">16</span>), primary_key=<span class="hljs-literal">True</span>),
)
metadata_obj.create_all(engine)

rows = [
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">1</span>, <span class="hljs-string">&quot;waiter_name&quot;</span>: <span class="hljs-string">&quot;Corey Johnson&quot;</span>},
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">2</span>, <span class="hljs-string">&quot;waiter_name&quot;</span>: <span class="hljs-string">&quot;Michael Watts&quot;</span>},
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">3</span>, <span class="hljs-string">&quot;waiter_name&quot;</span>: <span class="hljs-string">&quot;Michael Watts&quot;</span>},
    {<span class="hljs-string">&quot;receipt_id&quot;</span>: <span class="hljs-number">4</span>, <span class="hljs-string">&quot;waiter_name&quot;</span>: <span class="hljs-string">&quot;Margaret James&quot;</span>},
]
<span class="hljs-keyword">for</span> row <span class="hljs-keyword">in</span> rows:
    stmt = insert(receipts).values(**row)
    <span class="hljs-keyword">with</span> engine.begin() <span class="hljs-keyword">as</span> connection:
        cursor = connection.execute(stmt)`,wrap:!1}}),z=new J({props:{code:"dXBkYXRlZF9kZXNjcmlwdGlvbiUyMCUzRCUyMCUyMiUyMiUyMkFsbG93cyUyMHlvdSUyMHRvJTIwcGVyZm9ybSUyMFNRTCUyMHF1ZXJpZXMlMjBvbiUyMHRoZSUyMHRhYmxlLiUyMEJld2FyZSUyMHRoYXQlMjB0aGlzJTIwdG9vbCdzJTIwb3V0cHV0JTIwaXMlMjBhJTIwc3RyaW5nJTIwcmVwcmVzZW50YXRpb24lMjBvZiUyMHRoZSUyMGV4ZWN1dGlvbiUyMG91dHB1dC4lMEFJdCUyMGNhbiUyMHVzZSUyMHRoZSUyMGZvbGxvd2luZyUyMHRhYmxlcyUzQSUyMiUyMiUyMiUwQSUwQWluc3BlY3RvciUyMCUzRCUyMGluc3BlY3QoZW5naW5lKSUwQWZvciUyMHRhYmxlJTIwaW4lMjAlNUIlMjJyZWNlaXB0cyUyMiUyQyUyMCUyMndhaXRlcnMlMjIlNUQlM0ElMEElMjAlMjAlMjAlMjBjb2x1bW5zX2luZm8lMjAlM0QlMjAlNUIoY29sJTVCJTIybmFtZSUyMiU1RCUyQyUyMGNvbCU1QiUyMnR5cGUlMjIlNUQpJTIwZm9yJTIwY29sJTIwaW4lMjBpbnNwZWN0b3IuZ2V0X2NvbHVtbnModGFibGUpJTVEJTBBJTBBJTIwJTIwJTIwJTIwdGFibGVfZGVzY3JpcHRpb24lMjAlM0QlMjBmJTIyVGFibGUlMjAnJTdCdGFibGUlN0QnJTNBJTVDbiUyMiUwQSUwQSUyMCUyMCUyMCUyMHRhYmxlX2Rlc2NyaXB0aW9uJTIwJTJCJTNEJTIwJTIyQ29sdW1ucyUzQSU1Q24lMjIlMjAlMkIlMjAlMjIlNUNuJTIyLmpvaW4oJTVCZiUyMiUyMCUyMC0lMjAlN0JuYW1lJTdEJTNBJTIwJTdCY29sX3R5cGUlN0QlMjIlMjBmb3IlMjBuYW1lJTJDJTIwY29sX3R5cGUlMjBpbiUyMGNvbHVtbnNfaW5mbyU1RCklMEElMjAlMjAlMjAlMjB1cGRhdGVkX2Rlc2NyaXB0aW9uJTIwJTJCJTNEJTIwJTIyJTVDbiU1Q24lMjIlMjAlMkIlMjB0YWJsZV9kZXNjcmlwdGlvbiUwQSUwQXByaW50KHVwZGF0ZWRfZGVzY3JpcHRpb24p",highlighted:`<span class="hljs-meta">&gt;&gt;&gt; </span>updated_description = <span class="hljs-string">&quot;&quot;&quot;Allows you to perform SQL queries on the table. Beware that this tool&#x27;s output is a string representation of the execution output.
<span class="hljs-meta">... </span>It can use the following tables:&quot;&quot;&quot;</span>

<span class="hljs-meta">&gt;&gt;&gt; </span>inspector = inspect(engine)
<span class="hljs-meta">&gt;&gt;&gt; </span><span class="hljs-keyword">for</span> table <span class="hljs-keyword">in</span> [<span class="hljs-string">&quot;receipts&quot;</span>, <span class="hljs-string">&quot;waiters&quot;</span>]:
<span class="hljs-meta">... </span>    columns_info = [(col[<span class="hljs-string">&quot;name&quot;</span>], col[<span class="hljs-string">&quot;type&quot;</span>]) <span class="hljs-keyword">for</span> col <span class="hljs-keyword">in</span> inspector.get_columns(table)]

<span class="hljs-meta">... </span>    table_description = <span class="hljs-string">f&quot;Table &#x27;<span class="hljs-subst">{table}</span>&#x27;:\\n&quot;</span>

<span class="hljs-meta">... </span>    table_description += <span class="hljs-string">&quot;Columns:\\n&quot;</span> + <span class="hljs-string">&quot;\\n&quot;</span>.join([<span class="hljs-string">f&quot;  - <span class="hljs-subst">{name}</span>: <span class="hljs-subst">{col_type}</span>&quot;</span> <span class="hljs-keyword">for</span> name, col_type <span class="hljs-keyword">in</span> columns_info])
<span class="hljs-meta">... </span>    updated_description += <span class="hljs-string">&quot;\\n\\n&quot;</span> + table_description

<span class="hljs-meta">&gt;&gt;&gt; </span><span class="hljs-built_in">print</span>(updated_description)`,wrap:!1}}),K=new J({props:{code:"c3FsX3Rvb2wlMjAlM0QlMjBTUUxFeGVjdXRvclRvb2woKSUwQXNxbF90b29sLmRlc2NyaXB0aW9uJTIwJTNEJTIwdXBkYXRlZF9kZXNjcmlwdGlvbiUwQSUwQWFnZW50JTIwJTNEJTIwUmVhY3RDb2RlQWdlbnQoJTBBJTIwJTIwJTIwJTIwdG9vbHMlM0QlNUJzcWxfdG9vbCU1RCUyQyUwQSUyMCUyMCUyMCUyMGxsbV9lbmdpbmUlM0RIZkVuZ2luZSglMjJtZXRhLWxsYW1hJTJGTWV0YS1MbGFtYS0zLTcwQi1JbnN0cnVjdCUyMiklMkMlMEEpJTBBJTBBYWdlbnQucnVuKCUyMldoaWNoJTIwd2FpdGVyJTIwZ290JTIwbW9yZSUyMHRvdGFsJTIwbW9uZXklMjBmcm9tJTIwdGlwcyUzRiUyMik=",highlighted:`<span class="hljs-meta">&gt;&gt;&gt; </span>sql_tool = SQLExecutorTool()
<span class="hljs-meta">&gt;&gt;&gt; </span>sql_tool.description = updated_description

<span class="hljs-meta">&gt;&gt;&gt; </span>agent = ReactCodeAgent(
<span class="hljs-meta">... </span>    tools=[sql_tool],
<span class="hljs-meta">... </span>    llm_engine=HfEngine(<span class="hljs-string">&quot;meta-llama/Meta-Llama-3-70B-Instruct&quot;</span>),
<span class="hljs-meta">... </span>)

<span class="hljs-meta">&gt;&gt;&gt; </span>agent.run(<span class="hljs-string">&quot;Which waiter got more total money from tips?&quot;</span>)`,wrap:!1}}),tl=new At({props:{source:"https://github.com/huggingface/cookbook/blob/main/notebooks/en/agent_text_to_sql.md"}}),{c(){U=M("meta"),nl=n(),sl=M("p"),al=n(),i(T.$$.fragment),Ml=n(),i(w.$$.fragment),pl=n(),m=M("p"),m.innerHTML=Ll,ol=n(),h=M("p"),h.innerHTML=zl,il=n(),d=M("p"),d.textContent=Dl,yl=n(),b=M("p"),b.textContent=Kl,cl=n(),I=M("p"),I.innerHTML=Pl,jl=n(),g=M("p"),g.textContent=Ol,rl=n(),i(C.$$.fragment),ul=n(),i(f.$$.fragment),Jl=n(),i(A.$$.fragment),Ul=n(),B=M("p"),B.textContent=lt,Tl=n(),i(q.$$.fragment),wl=n(),W=M("pre"),W.textContent=tt,ml=n(),i(Q.$$.fragment),hl=n(),Z=M("p"),Z.textContent=st,dl=n(),E=M("p"),E.innerHTML=et,bl=n(),i(R.$$.fragment),Il=n(),N=M("pre"),N.textContent=nt,gl=n(),k=M("p"),k.innerHTML=at,Cl=n(),v=M("ul"),v.innerHTML=Mt,fl=n(),i(S.$$.fragment),Al=n(),V=M("p"),V.textContent=pt,Bl=n(),_=M("p"),_.innerHTML=ot,ql=n(),G=M("p"),G.innerHTML=it,Wl=n(),i($.$$.fragment),Ql=n(),i(x.$$.fragment),Zl=n(),i(X.$$.fragment),El=n(),H=M("p"),H.textContent=yt,Rl=n(),Y=M("p"),Y.innerHTML=ct,Nl=n(),i(F.$$.fragment),kl=n(),L=M("p"),L.innerHTML=jt,vl=n(),i(z.$$.fragment),Sl=n(),D=M("pre"),D.textContent=rt,Vl=n(),i(K.$$.fragment),_l=n(),P=M("pre"),P.textContent=ut,Gl=n(),O=M("p"),O.textContent=Jt,$l=n(),ll=M("p"),ll.textContent=Ut,xl=n(),i(tl.$$.fragment),Xl=n(),el=M("p"),this.h()},l(l){const t=gt("svelte-u9bgzb",document.head);U=p(t,"META",{name:!0,content:!0}),t.forEach(s),nl=a(l),sl=p(l,"P",{}),Tt(sl).forEach(s),al=a(l),y(T.$$.fragment,l),Ml=a(l),y(w.$$.fragment,l),pl=a(l),m=p(l,"P",{"data-svelte-h":!0}),o(m)!=="svelte-1xlqnsv"&&(m.innerHTML=Ll),ol=a(l),h=p(l,"P",{"data-svelte-h":!0}),o(h)!=="svelte-wem15p"&&(h.innerHTML=zl),il=a(l),d=p(l,"P",{"data-svelte-h":!0}),o(d)!=="svelte-3ojw25"&&(d.textContent=Dl),yl=a(l),b=p(l,"P",{"data-svelte-h":!0}),o(b)!=="svelte-1hw7g6s"&&(b.textContent=Kl),cl=a(l),I=p(l,"P",{"data-svelte-h":!0}),o(I)!=="svelte-q251yb"&&(I.innerHTML=Pl),jl=a(l),g=p(l,"P",{"data-svelte-h":!0}),o(g)!=="svelte-1tzrwc8"&&(g.textContent=Ol),rl=a(l),y(C.$$.fragment,l),ul=a(l),y(f.$$.fragment,l),Jl=a(l),y(A.$$.fragment,l),Ul=a(l),B=p(l,"P",{"data-svelte-h":!0}),o(B)!=="svelte-ec23hm"&&(B.textContent=lt),Tl=a(l),y(q.$$.fragment,l),wl=a(l),W=p(l,"PRE",{"data-svelte-h":!0}),o(W)!=="svelte-jbb28p"&&(W.textContent=tt),ml=a(l),y(Q.$$.fragment,l),hl=a(l),Z=p(l,"P",{"data-svelte-h":!0}),o(Z)!=="svelte-fdu5cw"&&(Z.textContent=st),dl=a(l),E=p(l,"P",{"data-svelte-h":!0}),o(E)!=="svelte-jme9kq"&&(E.innerHTML=et),bl=a(l),y(R.$$.fragment,l),Il=a(l),N=p(l,"PRE",{"data-svelte-h":!0}),o(N)!=="svelte-1pz02dc"&&(N.textContent=nt),gl=a(l),k=p(l,"P",{"data-svelte-h":!0}),o(k)!=="svelte-84s4c8"&&(k.innerHTML=at),Cl=a(l),v=p(l,"UL",{"data-svelte-h":!0}),o(v)!=="svelte-1l8h02j"&&(v.innerHTML=Mt),fl=a(l),y(S.$$.fragment,l),Al=a(l),V=p(l,"P",{"data-svelte-h":!0}),o(V)!=="svelte-1t4r7cu"&&(V.textContent=pt),Bl=a(l),_=p(l,"P",{"data-svelte-h":!0}),o(_)!=="svelte-12rc58c"&&(_.innerHTML=ot),ql=a(l),G=p(l,"P",{"data-svelte-h":!0}),o(G)!=="svelte-1efohjn"&&(G.innerHTML=it),Wl=a(l),y($.$$.fragment,l),Ql=a(l),y(x.$$.fragment,l),Zl=a(l),y(X.$$.fragment,l),El=a(l),H=p(l,"P",{"data-svelte-h":!0}),o(H)!=="svelte-nr3fl7"&&(H.textContent=yt),Rl=a(l),Y=p(l,"P",{"data-svelte-h":!0}),o(Y)!=="svelte-6lwjm6"&&(Y.innerHTML=ct),Nl=a(l),y(F.$$.fragment,l),kl=a(l),L=p(l,"P",{"data-svelte-h":!0}),o(L)!=="svelte-frhx8p"&&(L.innerHTML=jt),vl=a(l),y(z.$$.fragment,l),Sl=a(l),D=p(l,"PRE",{"data-svelte-h":!0}),o(D)!=="svelte-d75mf2"&&(D.textContent=rt),Vl=a(l),y(K.$$.fragment,l),_l=a(l),P=p(l,"PRE",{"data-svelte-h":!0}),o(P)!=="svelte-1fqzj2e"&&(P.textContent=ut),Gl=a(l),O=p(l,"P",{"data-svelte-h":!0}),o(O)!=="svelte-1umn140"&&(O.textContent=Jt),$l=a(l),ll=p(l,"P",{"data-svelte-h":!0}),o(ll)!=="svelte-bt1f1w"&&(ll.textContent=Ut),xl=a(l),y(tl.$$.fragment,l),Xl=a(l),el=p(l,"P",{}),Tt(el).forEach(s),this.h()},h(){wt(U,"name","hf:doc:metadata"),wt(U,"content",qt)},m(l,t){Ct(document.head,U),e(l,nl,t),e(l,sl,t),e(l,al,t),c(T,l,t),e(l,Ml,t),c(w,l,t),e(l,pl,t),e(l,m,t),e(l,ol,t),e(l,h,t),e(l,il,t),e(l,d,t),e(l,yl,t),e(l,b,t),e(l,cl,t),e(l,I,t),e(l,jl,t),e(l,g,t),e(l,rl,t),c(C,l,t),e(l,ul,t),c(f,l,t),e(l,Jl,t),c(A,l,t),e(l,Ul,t),e(l,B,t),e(l,Tl,t),c(q,l,t),e(l,wl,t),e(l,W,t),e(l,ml,t),c(Q,l,t),e(l,hl,t),e(l,Z,t),e(l,dl,t),e(l,E,t),e(l,bl,t),c(R,l,t),e(l,Il,t),e(l,N,t),e(l,gl,t),e(l,k,t),e(l,Cl,t),e(l,v,t),e(l,fl,t),c(S,l,t),e(l,Al,t),e(l,V,t),e(l,Bl,t),e(l,_,t),e(l,ql,t),e(l,G,t),e(l,Wl,t),c($,l,t),e(l,Ql,t),c(x,l,t),e(l,Zl,t),c(X,l,t),e(l,El,t),e(l,H,t),e(l,Rl,t),e(l,Y,t),e(l,Nl,t),c(F,l,t),e(l,kl,t),e(l,L,t),e(l,vl,t),c(z,l,t),e(l,Sl,t),e(l,D,t),e(l,Vl,t),c(K,l,t),e(l,_l,t),e(l,P,t),e(l,Gl,t),e(l,O,t),e(l,$l,t),e(l,ll,t),e(l,xl,t),c(tl,l,t),e(l,Xl,t),e(l,el,t),Hl=!0},p:ht,i(l){Hl||(j(T.$$.fragment,l),j(w.$$.fragment,l),j(C.$$.fragment,l),j(f.$$.fragment,l),j(A.$$.fragment,l),j(q.$$.fragment,l),j(Q.$$.fragment,l),j(R.$$.fragment,l),j(S.$$.fragment,l),j($.$$.fragment,l),j(x.$$.fragment,l),j(X.$$.fragment,l),j(F.$$.fragment,l),j(z.$$.fragment,l),j(K.$$.fragment,l),j(tl.$$.fragment,l),Hl=!0)},o(l){r(T.$$.fragment,l),r(w.$$.fragment,l),r(C.$$.fragment,l),r(f.$$.fragment,l),r(A.$$.fragment,l),r(q.$$.fragment,l),r(Q.$$.fragment,l),r(R.$$.fragment,l),r(S.$$.fragment,l),r($.$$.fragment,l),r(x.$$.fragment,l),r(X.$$.fragment,l),r(F.$$.fragment,l),r(z.$$.fragment,l),r(K.$$.fragment,l),r(tl.$$.fragment,l),Hl=!1},d(l){l&&(s(nl),s(sl),s(al),s(Ml),s(pl),s(m),s(ol),s(h),s(il),s(d),s(yl),s(b),s(cl),s(I),s(jl),s(g),s(rl),s(ul),s(Jl),s(Ul),s(B),s(Tl),s(wl),s(W),s(ml),s(hl),s(Z),s(dl),s(E),s(bl),s(Il),s(N),s(gl),s(k),s(Cl),s(v),s(fl),s(Al),s(V),s(Bl),s(_),s(ql),s(G),s(Wl),s(Ql),s(Zl),s(El),s(H),s(Rl),s(Y),s(Nl),s(kl),s(L),s(vl),s(Sl),s(D),s(Vl),s(_l),s(P),s(Gl),s(O),s($l),s(ll),s(xl),s(Xl),s(el)),s(U),u(T,l),u(w,l),u(C,l),u(f,l),u(A,l),u(q,l),u(Q,l),u(R,l),u(S,l),u($,l),u(x,l),u(X,l),u(F,l),u(z,l),u(K,l),u(tl,l)}}}const qt='{"title":"Agent for text-to-SQL with automatic error correction","local":"agent-for-text-to-sql-with-automatic-error-correction","sections":[{"title":"Setup SQL tables","local":"setup-sql-tables","sections":[],"depth":2},{"title":"Build our agent","local":"build-our-agent","sections":[],"depth":2},{"title":"Increasing difficulty: Table joins","local":"increasing-difficulty-table-joins","sections":[],"depth":2}],"depth":1}';function Wt(Fl){return dt(()=>{new URLSearchParams(window.location.search).get("fw")}),[]}class Nt extends bt{constructor(U){super(),It(this,U,Wt,Bt,mt,{})}}export{Nt as component};
