# Автоматическое распознавание речи с помощью конвейера

Автоматическое распознавание речи (ASR от "Automatic Speech Recognition") - это задача транскрибации (преобразование аудиозаписи речи в текст).
Эта задача имеет множество практических применений - от создания субтитров для видео до обеспечения голосовых команд для таких виртуальных
помощников, как Siri и Alexa.

В этом разделе мы используем конвейер `automatic-speech-recognition` для транскрибации аудиозаписи человека, задающего вопрос об
оплате счета, используя тот же набор данных MINDS-14, что и ранее.

Для начала загрузите набор данных и повысьте его частоту до 16 кГц, как описано в [Классификация аудио с помощью конвейера](audio_classification_pipeline),
если вы еще не сделали этого.

Для транскрибации аудиозаписи мы можем использовать конвейер `automatic-speech-recognition` из 🤗 Transformers. Давайте инстанцируем конвейер:

```python
from transformers import pipeline

asr = pipeline("automatic-speech-recognition")
```

Далее мы возьмем пример из набора данных и передадим его исходные данные в конвейер:

```python
example = minds[0]
asr(example["audio"]["array"])
```

**Output:**
```out
{"text": "I WOULD LIKE TO PAY MY ELECTRICITY BILL USING MY COD CAN YOU PLEASE ASSIST"}
```

Сравним этот вывод с реальной транскрипцией для данного примера:

```python
example["english_transcription"]
```

**Output:**
```out
"I would like to pay my electricity bill using my card can you please assist"
```

Похоже, что модель неплохо справилась с транскрибацией аудиозаписи! По сравнению с оригинальной транскрипцией она ошиблась только в одном слове
("card"), что довольно хорошо, учитывая австралийский акцент диктора, в котором буква "r" часто не произносится. Тем не менее, я бы не рекомендовал
пытаться оплатить свой следующий счет за электричество рыбой (cod в переводе с английского - "треска")!

По умолчанию в этом конвейере используется модель, обученная автоматическому распознаванию речи для английского языка, что вполне подходит для
данного примера. Если вы хотите попробовать транскрибировать другие подмножества MINDS-14 на другом языке, вы можете найти предварительно
обученную ASR-модель [на 🤗 Hugging Face Hub](https://huggingface.co/models?pipeline_tag=automatic-speech-recognition&language=fr&sort=downloads).
Список моделей можно отфильтровать сначала по задаче, затем по языку. Найдя нужную модель, передайте ее имя в качестве аргумента `model` в конвейер.

Попробуем это сделать для немецкого сплита MINDS-14. Загрузим подмножество "de-DE":

```python
from datasets import load_dataset
from datasets import Audio

minds = load_dataset("PolyAI/minds14", name="de-DE", split="train")
minds = minds.cast_column("audio", Audio(sampling_rate=16_000))
```

Возьмите пример и посмотрите, какой должна быть транскрибация:

```python
example = minds[0]
example["transcription"]
```

**Output:**
```out
"ich möchte gerne Geld auf mein Konto einzahlen"
```

Найдите на 🤗 Hugging Face Hub предварительно обученную ASR-модель для немецкого языка, инстанцируйте конвейер и транскрибируйте пример:

```py
from transformers import pipeline

asr = pipeline("automatic-speech-recognition", model="maxidl/wav2vec2-large-xlsr-german")
asr(example["audio"]["array"])
```

**Output:**
```out
{"text": "ich möchte gerne geld auf mein konto einzallen"}
```

При работе над решением собственной задачи начать с простого конвейера, подобного тем, что мы показали в этом блоке,
это ценный инструмент, дающий ряд преимуществ:
- может существовать предварительно обученная модель, которая уже очень хорошо решает вашу задачу, экономя ваше время;
- pipeline() берет на себя всю предварительную и последующую обработку, так что вам не нужно беспокоиться о том, чтобы
привести данные в нужный формат для модели;
- если результат не идеален, это все равно дает возможность быстро получить baseline для последующего дообучения модели;
- после того как вы дообучите модель на своих пользовательских данных и разместите ее на Hugging Face Hub, все сообщество сможет
быстро и легко использовать ее с помощью метода `pipeline()`, делая ИИ более доступным.

