import customtkinter as ctk

class PositionSelector(ctk.CTkToplevel):
    def __init__(self, button, char_list, v_large_font, pos_viewer, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.title("Set Character's Position")
        self.geometry("300x280")
        self.resizable(False, False)
        self.attributes('-topmost', True)

        self.position_window_at_mouse()
        
        # 입력으로 받은 버튼과 리스트 저장
        self.target_button = button
        self.char_list = char_list
        self.pos_viewer = pos_viewer
        
        # 선택된 좌표 저장 변수
        self.selected_pos = None
        
        # 메인 프레임 생성
        self.main_frame = ctk.CTkFrame(self)
        self.main_frame.pack(fill="both", expand=True, padx=10, pady=10)
        
        # 타이틀 레이블
        self.title_label = ctk.CTkLabel(
            self.main_frame, 
            text="Set Character Position", 
            font=v_large_font
        )
        self.title_label.pack(pady=(0, 20))
        
        # 그리드 프레임 생성
        self.grid_frame = ctk.CTkFrame(self.main_frame)
        self.grid_frame.pack(pady=10)
        
        # 버튼 그리드 생성 (5x5)
        self.grid_buttons = {}
        
        # 행 레이블 (A-E)
        row_labels = ['A', 'B', 'C', 'D', 'E']
        
        # 버튼 생성
        for col_idx, col_label in enumerate(row_labels):  # row_labels 변수명은 유지하되 의미상 열(column) 레이블
            for row_idx in range(5):
                row_num = row_idx + 1
                pos = f"{col_label}{row_num}"
                
                button = ctk.CTkButton(
                    self.grid_frame,
                    text=f"{col_label}{row_num}",
                    width=35,
                    height=30,
                    corner_radius=0,
                    fg_color="grey10",
                    hover_color="#505050",
                    command=lambda p=pos: self.select_position(p)
                )
                button.grid(row=row_idx, column=col_idx, padx=2, pady=2)
                
                self.grid_buttons[pos] = button
        
        # Done 버튼
        self.done_button = ctk.CTkButton(
            self.main_frame,
            text="Done",
            width=100,
            height=40,
            font=v_large_font,
            command=self.on_done
        )
        #self.done_button.pack(side="right", padx=10, pady=10)
    
    def select_position(self, position):
        # 이전에 선택된 버튼을 원래 색상으로 복원
        if self.selected_pos:
            self.grid_buttons[self.selected_pos].configure(fg_color="grey10")
        
        # 새로 선택된 버튼 색상 변경
        self.selected_pos = position
        self.on_done()
        #self.grid_buttons[position].configure(fg_color="#F4F4C3")
    
    def on_done(self):
        if self.selected_pos:
            # 버튼 텍스트 가져오기
            button_text = self.target_button.cget("text")
            
            # ": " 기준으로 분리
            if ": " in button_text:
                prefix, _ = button_text.split(" : ", 1)
                
                # 접두사에서 캐릭터 번호 추출
                char_num = int(prefix[-1])
                
                # 버튼 텍스트 업데이트
                self.target_button.configure(text=f"{prefix} : {self.selected_pos}")
                
                # 리스트 업데이트
                self.char_list[char_num - 1] = self.selected_pos
                
                # 선택된 버튼 색상 초기화
                self.grid_buttons[self.selected_pos].configure(fg_color="grey10")

                self.update_position_image()
                
                # 창 닫기
                self.withdraw()
            else:
                print("버튼 텍스트 형식이 잘못되었습니다.")
        else:
            print("위치를 선택해주세요.")

    def position_window_at_mouse(self):
        """마우스 위치에 창을 표시합니다."""
        # 현재 마우스 위치 가져오기
        mouse_x = self.winfo_pointerx()
        mouse_y = self.winfo_pointery()
        
        # 창 크기 가져오기
        window_width = 420
        window_height = 280
        
        # 화면 해상도 가져오기
        screen_width = self.winfo_screenwidth()
        screen_height = self.winfo_screenheight()
        
        # 창이 화면 밖으로 나가지 않도록 조정
        x_position = min(mouse_x, screen_width - window_width)
        y_position = min(mouse_y, screen_height - window_height)
        
        # 창의 위치 설정
        self.geometry(f"{window_width}x{window_height}+{x_position}+{y_position}")

    def update_position_image(self):
        from PIL import Image, ImageDraw
        
        colors = [
            (255, 182, 193),  # 연한 분홍색 (Light Pink)
            (173, 216, 230),  # 연한 파란색 (Light Blue)
            (144, 238, 144),  # 연한 녹색 (Light Green)
            (255, 255, 153),  # 연한 노란색 (Light Yellow)
            (216, 191, 216),  # 연한 보라색 (Light Purple)
            (250, 218, 185)   # 연한 주황색 (Light Orange)
        ]

        img = Image.new('RGB', (25, 25), "#4B4B4B")
        draw = ImageDraw.Draw(img)

        position_counts = {}
        
        for idx, pos in enumerate(self.char_list):
            if pos is not None:  # None이 아닌 위치만 처리
                col = ord(pos[0]) - ord('A')  # A=0, B=1, ..., E=4
                row = int(pos[1]) - 1  # 1=0, 2=1, ..., 5=4

                if pos in position_counts:
                    position_counts[pos] += 1
                    offset = position_counts[pos]
                else:
                    position_counts[pos] = 0
                    offset = 0

                x_offset = offset * 1
                y_offset = offset * 1

                x = col * 5 + x_offset
                y = row * 5 + y_offset

                if x < 21 and y < 21:
                    color = colors[idx % len(colors)]  
                    draw.rectangle([x, y, x+4, y+4], fill=color, outline=None)

        ctk_img = ctk.CTkImage(light_image=img, size=(50, 50))
        self.pos_viewer.configure(image=ctk_img)

        self.pos_viewer._image = ctk_img