import tkinter as tk
from tkinter import font
import customtkinter
import NAIA_utils, re
from danbooru_character import character_dict as dan_character_dict

def strip_brackets(keyword):
    prefix_match = re.match(r'^[\{\[\(]+', keyword)
    prefix = prefix_match.group(0) if prefix_match else ''
    suffix_match = re.search(r'[\}\]\)]+$', keyword)
    suffix = suffix_match.group(0) if suffix_match else ''
    stripped_keyword = keyword[len(prefix):len(keyword) - len(suffix)]
    return stripped_keyword, prefix, suffix

def restore_brackets(keyword, prefix, suffix):
    return f"{prefix}{keyword}{suffix}"

def update_processed_prompt(app_mode, widget, last_fixprompt, new_prompt_list):
    updated_texts = []
    updated_indices = []
    # 수정된 항목의 인덱스 추적
    for index, (new, old) in enumerate(zip(last_fixprompt, new_prompt_list)):
        if old.strip() != new.strip():
            if app_mode == "NAI": 
                updated_texts.append(new.strip())
            else:
                if not (new.startswith('(') and new.endswith(')')) and hasattr(widget, "search"):
                    updated_texts.append(new.strip().replace('(', '\\(').replace(')', '\\)'))
                else:
                    updated_texts.append(new.strip())
            updated_indices.append(index)
    if updated_indices:
        # Text 위젯인 경우 search, delete, insert 메서드를 사용함.
        # Entry 위젯은 전체 텍스트를 다시 설정하는 방식으로 처리합니다.
        if hasattr(widget, "search"):
            for idx in updated_indices:
                if idx + 1 != len(new_prompt_list):
                    # 마지막 항목이 아니라면, 토큰 뒤에 콤마를 붙여 검색
                    start_index = widget.search(new_prompt_list[idx] + ",", "1.0", stopindex="end")
                else:
                    # 마지막 항목이면, 위젯의 뒤에서부터 역방향으로 검색
                    start_index = widget.search(new_prompt_list[idx], "end", stopindex="1.0", backwards=True)
                if start_index:
                    end_index = f"{start_index} + {len(new_prompt_list[idx])} chars"
                    widget.delete(start_index, end_index)
                    if idx + 1 != len(new_prompt_list): widget.insert(start_index, " "+updated_texts[updated_indices.index(idx)])
                    else: widget.insert(start_index, " "+updated_texts[updated_indices.index(idx)])
        else:
            # Entry 위젯의 경우 전체 텍스트를 콤마로 join 하여 업데이트
            for idx in updated_indices:
                new_prompt_list[idx] = updated_texts[updated_indices.index(idx)]
            widget.delete(0, tk.END)
            widget.insert(0, ", ".join([s.strip() for s in new_prompt_list]))
    last_fixprompt[:] = [key.strip() for key in new_prompt_list]

class AutoCompleteHandler:
    def __init__(self, master, app_mode, text_widget, autocomplete_var, wildcard_dict_tree):
        """
        :param master: 팝업 생성 등에 사용할 부모 위젯 (예: 프레임 또는 루트)
        :param text_widget: 자동완성이 적용될 CTkTextbox(또는 tk.Text) 또는 CTkEntry(또는 tk.Entry) 위젯
        :param app_mode: 앱 모드 (예: "NAI" 등, update_processed_prompt에서 사용)
        :param autocomplete_var: 자동완성 사용 여부를 담은 tk.IntVar (1이면 사용)
        :param wildcard_dict_tree: NAIA_utils.find_top_matches에 전달할 사전
        """
        self.master = master
        self.text_widget = text_widget
        self.app_mode = app_mode
        self.autocomplete = autocomplete_var
        self.wildcard_dict_tree = wildcard_dict_tree

        # 위젯 타입에 따라 get/set 메서드를 정의
        if isinstance(text_widget, tk.Entry) or isinstance(text_widget, customtkinter.CTkEntry):
            self.is_entry = True
            self.get_text = lambda: self.text_widget.get()
            self.set_text = lambda txt: (self.text_widget.delete(0, tk.END), self.text_widget.insert(0, txt))
        else:
            self.is_entry = False
            self.get_text = lambda: self.text_widget.get("0.0", "end-1c")
            self.set_text = lambda txt: (self.text_widget.delete("1.0", tk.END), self.text_widget.insert("1.0", txt))

        # 자동완성 상태 변수
        self.last_prompt = []         # 콤마(,) 단위로 분리된 텍스트 리스트
        self.last_timer = None        # 타이머 변수

        # 3초 후에 위젯의 초기 프롬프트를 저장
        self.text_widget.after(3000, self.lfpi_fix)

        # 자동완성 팝업창과 리스트박스 생성
        self.popup = tk.Toplevel(master)
        self.popup.withdraw()
        self.popup.overrideredirect(True)
        self.listbox = tk.Listbox(
            self.popup,
            font=font.Font(family='Pretendard', size=13),
            bg='#2B2B2B',
            fg='#F8F8F8',
            borderwidth=2,
            highlightbackground='lightgrey',
            width=50,
            height=11
        )
        self.hint_label = tk.Label(
            self.popup,
            text="ESC를 누르거나, 다른곳을 누르면 팝업창이 닫힙니다",
            bg="white",
            fg="black",
            font=font.Font(family='Pretendard', size=11)
        )
        self.listbox.pack()
        self.hint_label.pack(side=tk.BOTTOM, fill=tk.X, padx=5, pady=2)
        self.popup.bind("<Escape>", lambda e: self.popup.withdraw())

        # 팝업이 포커스를 잃으면 숨김
        self.popup.bind("<FocusOut>", self.popup_focus_out)
        self.target_index = None      # 변경 대상 인덱스

        # 리스트박스 선택 시 자동완성 항목 삽입 처리
        self.listbox.bind("<<ListboxSelect>>", self.hard_insertion)

        # 입력된 요소 관련 추가 변수들
        self.last_element = ""
        self.last_element_prefix = ""
        self.last_element_suffix = ""

        # 텍스트 위젯(또는 Entry) 이벤트 바인딩
        self.text_widget.bind("<KeyRelease>", self.key_yield)
        self.text_widget.bind("<FocusOut>", self.popup_focus_out)

    def lfpi_fix(self):
        """위젯의 현재 내용을 콤마(,)로 분리하여 last_prompt에 저장"""
        self.last_prompt = [key for key in self.get_text().split(',')]
        #print("[lfpi_fix] Initial last_prompt set:", self.last_prompt)

    def popup_focus_out(self, event=None):
        """팝업이 포커스를 잃으면 숨깁니다."""
        #print("[popup_focus_out] Popup losing focus; hiding popup.")
        self.popup.withdraw()

    def hard_insertion(self, event=None):
        """
        리스트박스에서 항목을 선택하면 호출됩니다.
        선택된 항목을 현재 위젯의 대상 요소로 반영합니다.
        """
        try:
            selected_indices = self.listbox.curselection()
            keyword = self.listbox.get(selected_indices[0])
            index = keyword.rfind("  ")
            keyword = keyword[index + 2:]
            keyword = re.sub(r'^[0-9]+:', '', keyword)
            if "from:" in self.last_element and self.last_element.count(":") == 2:
                pass
            elif "from:" in self.last_element:
                keyword = "from:" + keyword + ":"
            elif "character:" in self.last_element:
                keyword = keyword + ", " + dan_character_dict.get(keyword, "")
            elif "wildcard:" in self.last_element:
                keyword = "__" + keyword + "__"
            self.popup.withdraw()
            keyword = restore_brackets(keyword, self.last_element_prefix, self.last_element_suffix)
            self.last_element_prefix = ''
            self.last_element_suffix = ''
            self.last_prompt[self.target_index] = keyword.strip()
            current_prompt = [key for key in self.get_text().split(',')]
            update_processed_prompt(self.app_mode, self.text_widget, self.last_prompt, current_prompt)
            #print("[hard_insertion] Inserted keyword:", keyword)
            if "from:" in self.last_element and self.last_element.count(":") == 1:
                self.last_prompt[self.target_index] = keyword[:-1]
                self.target_index = None
                self.last_element = ""
                self.text_widget.after(900, lambda: self.key_yield(None))
            else:
                self.target_index = None
                self.last_element = ""
        except Exception as e:
            #print("[hard_insertion] Exception:", e)
            pass

    def delayed_key_yield(self):
        """키 입력 후 200ms 후에 호출되어 추천 리스트를 갱신합니다."""
        #print("[delayed_key_yield] Triggered. Current text:", self.get_text())
        
        current_prompt = [key.strip() for key in self.get_text().split(',')]

        try:
            if self.last_prompt != current_prompt:
                self.listbox.delete(0, tk.END)
        except Exception:
            pass

        changed_elements = []
        changed_indices = []
        
        # 리스트 길이 차이를 먼저 체크하여 새로운 항목이 추가된 경우 처리
        if len(current_prompt) != len(self.last_prompt):
            changed_elements.append(current_prompt[-1])
            changed_indices.append(len(current_prompt) - 1)
        else:
            for index, (old, new) in enumerate(zip(self.last_prompt, current_prompt)):
                if old.strip() != new.strip():
                    changed_elements.append(new)
                    changed_indices.append(index)
        
        #print("[delayed_key_yield] Changed elements:", changed_elements)
        
        if changed_elements and changed_elements[-1].strip() != '':
            target_element = changed_elements[-1]
            self.last_element = target_element
            if target_element.strip() in [k.strip() for k in self.last_prompt]:
                self.last_prompt = [key for key in current_prompt]
                #print("[delayed_key_yield] No significant change detected; returning.")
                return
            target_index = changed_indices[-1]
            self.target_index = target_index
            stripped_target, prefix, suffix = strip_brackets(target_element.strip())
            self.last_element = target_element.strip()
            self.last_element_prefix = prefix
            self.last_element_suffix = suffix
            #print(f"[delayed_key_yield] Target element: '{target_element.strip()}', Stripped: '{stripped_target}', Prefix: '{prefix}', Suffix: '{suffix}'")
            if hasattr(self.master, 'wildcard_dict_tree'):
                _list = NAIA_utils.find_top_matches(stripped_target, self.master.wildcard_dict_tree)
            else:
                _list = NAIA_utils.find_top_matches(stripped_target, self.wildcard_dict_tree)
            if _list:
                for k, v in _list:
                    k_formatted = k
                    v_formatted = f"{v:>9}".replace(' ', '  ')
                    formatted_line = f"{v_formatted}  {k_formatted}"
                    self.listbox.insert(tk.END, formatted_line)
                # 팝업 위치 계산: Entry 위젯은 dlineinfo가 없으므로 따로 처리
                if self.is_entry:
                    abs_x = self.text_widget.winfo_rootx()
                    abs_y = self.text_widget.winfo_rooty() + self.text_widget.winfo_height()
                else:
                    bbox = self.text_widget.dlineinfo("insert")
                    if not bbox:
                        #print("[delayed_key_yield] dlineinfo('insert') returned None, scheduling retry in 100ms.")
                        self.text_widget.after(100, self.delayed_key_yield)
                        return
                    x, y, width, height, baseline = bbox
                    abs_x = self.text_widget.winfo_rootx() + x
                    abs_y = self.text_widget.winfo_rooty() + y + height * 2
                self.popup.geometry(f"+{abs_x}+{abs_y}")
                self.popup.deiconify()
                #print(f"[delayed_key_yield] Popup shown at ({abs_x}, {abs_y}).")
            else:
                self.popup.withdraw()
                #print("[delayed_key_yield] No matches found; hiding popup.")
        else:
            pass
            #self.popup.withdraw()
            #print("[delayed_key_yield] No changed elements or target element empty; hiding popup.")
        
        self.last_prompt = []
        self.last_prompt[:] = current_prompt
        #print("[delayed_key_yield] Updated last_prompt:", self.last_prompt)

    def key_yield(self, event):
        """키 입력 시 200ms 후에 delayed_key_yield를 실행합니다."""
        #print("[key_yield] Key event triggered. Current text:", self.get_text())
        if self.autocomplete.get() == 1:
            if self.last_timer:
                #print("[key_yield] Cancelling previous timer.")
                self.text_widget.after_cancel(self.last_timer)
            self.last_timer = self.text_widget.after(100, self.delayed_key_yield)
            #print("[key_yield] New timer set.")
